var Microsoft;
(function (Microsoft) {
    var VisualStudio;
    (function (VisualStudio) {
        var DiagnosticsHub;
        (function (DiagnosticsHub) {
            var Controllers;
            (function (Controllers) {
                "use strict";
                var ViewController = (function () {
                    function ViewController() {
                        this._initialProgressConst = 0.3;
                        this._maxProgressConst = 1.0;
                        var progressMain = document.querySelector(".progress-main");
                        this._progressControl = progressMain.querySelector(".progress-bar");
                        this._progressText = progressMain.querySelector(".progress-text");
                        this._progressText.innerText = Microsoft.Plugin.Resources.getString("/DiagnosticsHubResources/DocumentStatusView_ProcessingData");
                        this._toolsContainer = document.querySelector(".progress-tools");
                        this._logger = Microsoft.VisualStudio.DiagnosticsHub.getLogger();
                        this._document = Microsoft.VisualStudio.DiagnosticsHub.getCurrentDocument();
                        this._javaScriptJmcHelper = new Controllers.JavaScriptJmc();
                        this._solutionService = new Controllers.SolutionService();
                        this._analyzersProgressScale = this._maxProgressConst - this._initialProgressConst;
                        this._marshaler = Microsoft.Plugin.Utilities.JSONMarshaler.attachToPublishedObject("Microsoft.DiagnosticsHub.VisualStudio.Presentation.JavaScriptModels.DocumentStatusViewMarshaler", {}, true);
                    }
                    ViewController.prototype.initialize = function () {
                        var progressTools = [];
                        var analyzerIdsToInitialize = [];
                        Microsoft.VisualStudio.DiagnosticsHub.DataWarehouse.loadDataWarehouse()
                            .then(this.processDataWarehouseConfiguration.bind(this, analyzerIdsToInitialize))
                            .then(this.getToolInformation.bind(this))
                            .then(this.processToolInformation.bind(this, progressTools, analyzerIdsToInitialize))
                            .then(this.beginDataWarehouseInitialization.bind(this))
                            .then(this.pushInitialDataSourcesToDataWarehouse.bind(this))
                            .then(this.addAllExecutableCodeOutputsToJmc.bind(this), this.onError.bind(this), this.onProgress.bind(this, progressTools))
                            .then(this.addJavaScriptToJmc.bind(this))
                            .then(this.initializeDataWarehouseComplete.bind(this))
                            .done(this.initializeAnalyzers.bind(this, analyzerIdsToInitialize, progressTools));
                    };
                    ViewController.prototype.processDataWarehouseConfiguration = function (analyzerIdsToInitialize, dw) {
                        this._dataWarehouse = dw;
                        var configuration = this._dataWarehouse.getConfiguration();
                        if (configuration.analyzers) {
                            for (var i = 0; i < configuration.analyzers.length; i++) {
                                var analyzer = configuration.analyzers[i];
                                if (analyzer.initialize) {
                                    analyzerIdsToInitialize.push(analyzer.clsid);
                                }
                            }
                        }
                        return Microsoft.Plugin.Promise.wrap(null);
                    };
                    ViewController.prototype.getToolInformation = function () {
                        return this._document.getTools();
                    };
                    ViewController.prototype.processToolInformation = function (progressTools, analyzerIdsToInitialize, tools) {
                        tools.sort(function (a, b) {
                            return a.name.toLocaleUpperCase().localeCompare(b.name.toLocaleUpperCase());
                        });
                        var ulElement = document.createElement("ul");
                        this._toolsContainer.appendChild(ulElement);
                        for (var i = 0; i < tools.length; i++) {
                            var currTool = tools[i];
                            var requireInitialization = analyzerIdsToInitialize.some(function (analyzerId) {
                                return this.toolContainsAnalyzer(currTool, analyzerId);
                            }, this);
                            var liElement = document.createElement("li");
                            liElement.className = "progress-line";
                            ulElement.appendChild(liElement);
                            var spanElement = document.createElement("span");
                            spanElement.className = "progress-text";
                            spanElement.innerText = currTool.name;
                            liElement.appendChild(spanElement);
                            var progressBarElement = document.createElement("progress");
                            progressBarElement.className = "progress-bar";
                            progressBarElement.max = this._maxProgressConst;
                            progressBarElement.value = 0.0;
                            liElement.appendChild(progressBarElement);
                            progressTools.push({
                                tool: currTool,
                                progressBar: progressBarElement,
                                span: spanElement,
                                requireInitialization: requireInitialization
                            });
                        }
                        return Microsoft.Plugin.Promise.wrap(null);
                    };
                    ViewController.prototype.beginDataWarehouseInitialization = function () {
                        this._logger.debug("Requesting data warehouse initialization");
                        return this._dataWarehouse.beginInitialization();
                    };
                    ViewController.prototype.pushInitialDataSourcesToDataWarehouse = function () {
                        var configuration = this._dataWarehouse.getConfiguration();
                        if (configuration && configuration.dataSources) {
                            this._logger.debug("Pushing initial data source(s) to data warehouse");
                            return this._dataWarehouse.pushDataSources(configuration.dataSources);
                        }
                        return Microsoft.Plugin.Promise.wrap(null);
                    };
                    ViewController.prototype.initializeDataWarehouseComplete = function () {
                        this._logger.debug("Indicating data warehouse initialization complete");
                        return this._dataWarehouse.endInitialization();
                    };
                    ViewController.prototype.initializeAnalyzers = function (analyzerIdsToInitialize, progressTools) {
                        var _this = this;
                        var inProgress = analyzerIdsToInitialize.length;
                        var errors = [];
                        var verifyProgress = function () {
                            if (inProgress === 0) {
                                if (errors.length === 0) {
                                    _this._logger.debug("Got DataWarehouse. Analyzing finished.");
                                    _this._marshaler._call("analyzingFinished");
                                }
                                else {
                                    _this._logger.error(JSON.stringify(errors[0]));
                                    _this._marshaler._call("analyzingFailed");
                                }
                            }
                        };
                        if (inProgress > 0) {
                            var updateProgress = function (clsid, progress) {
                                for (var i = 0; i < progressTools.length; i++) {
                                    if (progressTools[i].requireInitialization) {
                                        if (_this.toolContainsAnalyzer(progressTools[i].tool, clsid)) {
                                            if (progress.finished) {
                                                progressTools[i].progressBar.value = progressTools[i].progressBar.max;
                                            }
                                            else {
                                                var currentProgress = Math.min(progressTools[i].progressBar.max, _this._initialProgressConst + (_this._analyzersProgressScale * (progress.progressValue / progress.maxValue)));
                                                progressTools[i].progressBar.value = Math.max(progressTools[i].progressBar.value, currentProgress);
                                            }
                                        }
                                    }
                                }
                            };
                            var toolIds = progressTools.map(function (progressTool) {
                                return progressTool.tool.id;
                            });
                            var dhContextData = {
                                customDomain: {
                                    DiagnosticsHub_Initialization: "true",
                                    DiagnosticsHub_Tools: JSON.stringify(toolIds)
                                }
                            };
                            var preloadAnalyzer = function (clsid) {
                                _this._dataWarehouse.getFilteredData(dhContextData, clsid)
                                    .then(function () {
                                    inProgress--;
                                    verifyProgress();
                                }, function (error) {
                                    inProgress--;
                                    errors.push(error);
                                    verifyProgress();
                                }, function (progress) {
                                    updateProgress(clsid, progress);
                                });
                            };
                            for (var i = 0; i < analyzerIdsToInitialize.length; i++) {
                                preloadAnalyzer(analyzerIdsToInitialize[i]);
                            }
                        }
                        else {
                            verifyProgress();
                        }
                    };
                    ViewController.prototype.toolContainsAnalyzer = function (tool, analyzerId) {
                        if (tool && tool.dataWarehouse && tool.dataWarehouse.analyzers) {
                            return tool.dataWarehouse.analyzers.some(function (analyzer) {
                                return analyzer.id === analyzerId;
                            });
                        }
                        return false;
                    };
                    ViewController.prototype.addAllExecutableCodeOutputsToJmc = function () {
                        var _this = this;
                        this._logger.debug("Get all executable Solution outputs");
                        return this._solutionService.getAllExecutableCodeOutputs(false)
                            .then(function (executableOutputs) {
                            if (!executableOutputs || executableOutputs.length === 0) {
                                return Microsoft.Plugin.Promise.wrap(null);
                            }
                            var executableOutputPatterns = executableOutputs.concat(executableOutputs.map(function (output) { return "*\\" + output + ".*"; }));
                            return _this._dataWarehouse.setPrivateData(2, executableOutputPatterns);
                        });
                    };
                    ViewController.prototype.addJavaScriptToJmc = function () {
                        var _this = this;
                        this._logger.debug("Get JavaScript JMC information");
                        return this._dataWarehouse.getPrivateData(1)
                            .then(function (jsUrls) {
                            return _this._javaScriptJmcHelper.getJmcTypeForUrls(jsUrls)
                                .then(function (jmcTypes) {
                                if (jsUrls.length !== jmcTypes.length) {
                                    _this._logger.error("Returned jmcType array did not match url length");
                                    return Microsoft.Plugin.Promise.wrap(null);
                                }
                                var urlsWithJmcType = {};
                                for (var i = 0; i < jmcTypes.length; i++) {
                                    urlsWithJmcType[jsUrls[i]] = jmcTypes[i];
                                }
                                return _this._dataWarehouse.setPrivateData(1, urlsWithJmcType);
                            });
                        });
                    };
                    ViewController.prototype.onError = function (error) {
                        this._logger.error(JSON.stringify(error));
                        this._marshaler._call("analyzingFailed");
                    };
                    ViewController.prototype.onProgress = function (progressTools, progress) {
                        this._progressControl.max = progress.maxValue;
                        this._progressControl.value = progress.finished ? progress.maxValue : progress.progressValue;
                        for (var i = 0; i < progressTools.length; i++) {
                            if (progressTools[i].requireInitialization) {
                                progressTools[i].progressBar.value = this._initialProgressConst * (this._progressControl.value / this._progressControl.max);
                            }
                            else {
                                progressTools[i].progressBar.value = progressTools[i].progressBar.max * (this._progressControl.value / this._progressControl.max);
                            }
                        }
                    };
                    return ViewController;
                }());
                var viewController;
                Microsoft.Plugin.addEventListener("pluginready", function () {
                    this.viewController = new ViewController();
                    this.viewController.initialize();
                });
            })(Controllers = DiagnosticsHub.Controllers || (DiagnosticsHub.Controllers = {}));
        })(DiagnosticsHub = VisualStudio.DiagnosticsHub || (VisualStudio.DiagnosticsHub = {}));
    })(VisualStudio = Microsoft.VisualStudio || (Microsoft.VisualStudio = {}));
})(Microsoft || (Microsoft = {}));

// SIG // Begin signature block
// SIG // MIIkJQYJKoZIhvcNAQcCoIIkFjCCJBICAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // aWf3renNuWDU7QGXaDRLQrSCIubxk1PU/su7MQtG7I+g
// SIG // gg2TMIIGETCCA/mgAwIBAgITMwAAAI6HkaRXGl/KPgAA
// SIG // AAAAjjANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDExMB4XDTE2MTExNzIyMDkyMVoX
// SIG // DTE4MDIxNzIyMDkyMVowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBANCH1EIrfp3ZxnrUosPjFZLS
// SIG // U52VF8lSNvpUv6sQr+nJ58wmU8PCc79t1gDlANzpamc0
// SIG // MPYWF7QBpZV8i7mkLOaLO3n2Iwx5j/NS30ABHMLGA53r
// SIG // Wc9z6dhxOZvwziVZLdLJWwrvftYyDl10EgTsngRTpmsC
// SIG // Z/hNWYt34Csh4O/ApEUSzwN7A8Y5w9Qi3FVcd0L/nLLl
// SIG // VWdoui12an9mU0fVRwrMON6Ne5cZfYLQJviljuWh8F5k
// SIG // EOT56yfG8uAI0A3yZ8DY8i/7idoV+a4PPgCXB9ELPnDU
// SIG // d6tyeEGYB7gXzKKxX+y981Bno9eU8NKLVY9TppWT5rJm
// SIG // z8k3aORjx88CAwEAAaOCAYAwggF8MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjdMCAEGCCsGAQUFBwMDMB0GA1UdDgQWBBSr
// SIG // yNbtshXSqo7xzO1sOPdFStCKuzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwMDEy
// SIG // K2IwNTBjNmU3LTc2NDEtNDQxZi1iYzRhLTQzNDgxZTQx
// SIG // NWQwODAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzcitW2o
// SIG // ynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEGCCsGAQUF
// SIG // BwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3J0MAwGA1UdEwEB
// SIG // /wQCMAAwDQYJKoZIhvcNAQELBQADggIBAESJAqxpU/PE
// SIG // trvUjGBT58psqElpZr6lmkGZOtid0lcCUWr6v5uW26Ym
// SIG // fQlW6NztJXV6pUdSqB5LFlPz7g+awwSVKcGChKRWMfyg
// SIG // ipGVtb9azqkBH2RGoebK8dd0e7+SCFFefDMCXlE7m+XY
// SIG // Ll8CTAmcGkPace3k2eei2nQsF63lDLUY9VQJ1L4cc80g
// SIG // e6T6yNvY2zqu+pDFo72VZa5GLVcpWNaS8GzaY/GPM6J+
// SIG // OHZe3fM17ayaO2KB0E4ZfEh8sAuPOMwtvNU5ZamVwQPi
// SIG // ksm5q9JXCqrcUgsuViej4piXV468qVluJJKOguIJc4LZ
// SIG // NYPMn3/RBI6IuOKag1iw1JrmMfqUR459puJOefPY02oz
// SIG // FlBw8UK7mAnp/8yVVVsIv5JSqAjE8ejx/0DX+Zo2nf26
// SIG // kIXSVT5QrUYf7yUMuJ46SARj73iYol0DDQLY3CCr5la1
// SIG // 3u8WZsPXVYIeT4J4yZ5UGhBgtxerQBORrrAZwZozne4y
// SIG // cs1lzE9GmC0PUWAefPv+2+gHeQf3oTM4/gma2497tjq9
// SIG // hYa4zLx9ATC3ex2pXRu9zE0X925HM9VA32rKLlG4tbnP
// SIG // wwTTO+Xj6RCM66e63qQuM2opLxRK6h7BIjg1BYXvwgQA
// SIG // DWvB2JYUSBWvflKwuGDEUrVKgreFKgBJKiaDJ1pB3r3V
// SIG // Zkm8C5x4cAm8MIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAA
// SIG // AzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2Vy
// SIG // dGlmaWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4
// SIG // MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29k
// SIG // ZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00
// SIG // uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03a8YS2Avw
// SIG // OMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+b
// SIG // U7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQ
// SIG // z7NEt13YxC4Ddato88tt8zpcoRb0RrrgOGSsbmQ1eKag
// SIG // Yw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAf
// SIG // TVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+E
// SIG // GvKhL1nkkDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVO
// SIG // VpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSH
// SIG // vMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rh
// SIG // KEmdX4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3
// SIG // s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
// SIG // w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
// SIG // sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ec
// SIG // XL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90lfdu
// SIG // +HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaI
// SIG // jAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEA
// SIG // MB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2oynUClTAZ
// SIG // BgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8E
// SIG // BAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAW
// SIG // gBRyLToCMZBDuRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBR
// SIG // ME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
// SIG // cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDEx
// SIG // XzIwMTFfMDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBO
// SIG // BggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIw
// SIG // MTFfMDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsG
// SIG // AQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
// SIG // cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBn
// SIG // AGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABlAG0A
// SIG // ZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oal
// SIG // mOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74
// SIG // w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11lhJB9i0ZQ
// SIG // VdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeU
// SIG // OeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb
// SIG // 7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0wI/zRive/DvQ
// SIG // vTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLw
// SIG // xS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn
// SIG // +N4sOiBpmLJZiWhub6e3dMNABQamASooPoI/E01mC8Cz
// SIG // TfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jy
// SIG // FqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw
// SIG // 3MYbBL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtI
// SIG // EJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
// SIG // 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
// SIG // gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF6
// SIG // 70EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr/Xmf
// SIG // wb1tbWrJUnMTDXpQzTGCFeowghXmAgEBMIGVMH4xCzAJ
// SIG // BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
// SIG // DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
// SIG // ZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29m
// SIG // dCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAACOh5Gk
// SIG // Vxpfyj4AAAAAAI4wDQYJYIZIAWUDBAIBBQCggdYwGQYJ
// SIG // KoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
// SIG // NwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkE
// SIG // MSIEIFgYYFP3M+uY7DRxSk6ogJZipePfSTSPaFqx5B4D
// SIG // dglzMGoGCisGAQQBgjcCAQwxXDBaoECAPgBEAG8AYwB1
// SIG // AG0AZQBuAHQAUwB0AGEAdAB1AHMAVgBpAGUAdwBDAG8A
// SIG // bgB0AHIAbwBsAGwAZQByAC4AagBzoRaAFGh0dHA6Ly9t
// SIG // aWNyb3NvZnQuY29tMA0GCSqGSIb3DQEBAQUABIIBABXQ
// SIG // HTNvaztjB/cJnwUFM+Aeqy/bTvqd2P0pgY7lL6jpMPhz
// SIG // faADT8svoEzf+KdOUVU2nJLbQZNPvSIU4fO2gCzXAU9k
// SIG // k5HHZg+xXYdju55dl81LGpr08ZX8cDKfYnL1RHg9rAXp
// SIG // cBMPo8fQrvINClo9cwvfFfqM4F0vP864sFgEGs8rTItP
// SIG // QOfHmZbqlS2Ezq+u0tiK8lDYU+/dJ14A0/5BIQhDJhpp
// SIG // vJfVb4A6rPUdcnxvYFNvMaut+L290k8WhwNl/VEzsMz5
// SIG // JLmx8VyWlFKfLolwiRJaP5aOYqaL9gPW9W4kxXZLPjnE
// SIG // 0Te7zX0ASYEGjYSkQVdzBobS53/zM/ChghNMMIITSAYK
// SIG // KwYBBAGCNwMDATGCEzgwghM0BgkqhkiG9w0BBwKgghMl
// SIG // MIITIQIBAzEPMA0GCWCGSAFlAwQCAQUAMIIBPAYLKoZI
// SIG // hvcNAQkQAQSgggErBIIBJzCCASMCAQEGCisGAQQBhFkK
// SIG // AwEwMTANBglghkgBZQMEAgEFAAQgxUwhHJoO9HRyrMEa
// SIG // IiAkAklv84eqQXG0j7Vf8oiM5roCBliLkcNgixgSMjAx
// SIG // NzAyMDkwMzQzMzYuODVaMAcCAQGAAgH0oIG5pIG2MIGz
// SIG // MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
// SIG // bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
// SIG // cm9zb2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BS
// SIG // MScwJQYDVQQLEx5uQ2lwaGVyIERTRSBFU046QjFCNy1G
// SIG // NjdGLUZFQzIxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1l
// SIG // LVN0YW1wIFNlcnZpY2Wggg7QMIIGcTCCBFmgAwIBAgIK
// SIG // YQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkG
// SIG // A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
// SIG // BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
// SIG // dCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0
// SIG // IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAw
// SIG // HhcNMTAwNzAxMjEzNjU1WhcNMjUwNzAxMjE0NjU1WjB8
// SIG // MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
// SIG // bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
// SIG // cm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNy
// SIG // b3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6
// SIG // IOz8E5f1+n9plGt0VBDVpQoAgoX77XxoSyxfxcPlYcJ2
// SIG // tz5mK1vwFVMnBDEfQRsalR3OCROOfGEwWbEwRA/xYIiE
// SIG // VEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJY
// SIG // YEbyWEeGMoQedGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5
// SIG // hoC732H8RsEnHSRnEnIaIYqvS2SJUGKxXf13Hz3wV3Ws
// SIG // vYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noq
// SIG // CjHw2k4GkbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJ
// SIG // k3jN/LzAyURdXhacAQVPIk0CAwEAAaOCAeYwggHiMBAG
// SIG // CSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ
// SIG // 80N7fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMA
// SIG // dQBiAEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUw
// SIG // AwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvX
// SIG // zpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3Js
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9N
// SIG // aWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYB
// SIG // BQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0Nl
// SIG // ckF1dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0gAQH/BIGV
// SIG // MIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEW
// SIG // MWh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9j
// SIG // cy9DUFMvZGVmYXVsdC5odG0wQAYIKwYBBQUHAgIwNB4y
// SIG // IB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQA
// SIG // YQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQAD
// SIG // ggIBAAfmiFEN4sbgmD+BcQM9naOhIW+z66bM9TG+zwXi
// SIG // qf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtR
// SIG // gkQS+7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3Tv
// SIG // QhDIr79/xn/yN31aPxzymXlKkVIArzgPF/UveYFl2am1
// SIG // a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon
// SIG // /VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em
// SIG // 4jnsGUpxY517IW3DnKOiPPp/fZZqkHimbdLhnPkd/DjY
// SIG // lPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/fmNZ
// SIG // JQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99
// SIG // g/DhO3EJ3110mCIIYdqwUB5vvfHhAN/nMQekkzr3ZUd4
// SIG // 6PioSKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0cs0d9LiF
// SIG // AR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXA
// SIG // L1QnIffIrE7aKLixqduWsqdCosnPGUFN4Ib5KpqjEWYw
// SIG // 07t0MkvfY3v1mYovG8chr1m1rtxEPJdQcdeh0sVV42ne
// SIG // V8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrv
// SIG // CScc1bN+NR4Iuto229Nfj950iEkSMIIE2jCCA8KgAwIB
// SIG // AgITMwAAALFxE3nfdfY1yAAAAAAAsTANBgkqhkiG9w0B
// SIG // AQsFADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2Fz
// SIG // aGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
// SIG // ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQD
// SIG // Ex1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDAe
// SIG // Fw0xNjA5MDcxNzU2NTdaFw0xODA5MDcxNzU2NTdaMIGz
// SIG // MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
// SIG // bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
// SIG // cm9zb2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BS
// SIG // MScwJQYDVQQLEx5uQ2lwaGVyIERTRSBFU046QjFCNy1G
// SIG // NjdGLUZFQzIxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1l
// SIG // LVN0YW1wIFNlcnZpY2UwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCqpCSUbVjWW7yhvQ/t166a5Gfg
// SIG // m9GLYYSuYr3i+BudY+Z3SP/1qsDvnf0cPV2kbW6Fhuac
// SIG // DFz6qy68wzR+kS+21MriVlJTuyzmsl9aZsWf8nyRIYjw
// SIG // r2IFoHqFCQm4hfiyL2mk2v1Hehkjcdsn/JGQpQ+TiGjO
// SIG // ljoKR6FFzT9l+7q1CLKojuYKVdhlNePD6suyHf+B0G9g
// SIG // N3fzMUGWVp/7e6KYpCBRNcaNsp+plmTe0RTeJtZU9TEC
// SIG // abGUbexZOVeZTfV8LD/pNXMaDbnWWr5Djo6Nt4f28HZM
// SIG // 5yoSyjg1qLcnUJ0wBhR2V6VVW2BB0jH9z7ke+vDRjpbu
// SIG // 4YEBadbnAgMBAAGjggEbMIIBFzAdBgNVHQ4EFgQUTlc9
// SIG // 94suFEtXsvwiXtPPtydEEDswHwYDVR0jBBgwFoAU1WM6
// SIG // XIoxkPNDe3xGG8UzaFqFbVUwVgYDVR0fBE8wTTBLoEmg
// SIG // R4ZFaHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9j
// SIG // cmwvcHJvZHVjdHMvTWljVGltU3RhUENBXzIwMTAtMDct
// SIG // MDEuY3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcw
// SIG // AoY+aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9j
// SIG // ZXJ0cy9NaWNUaW1TdGFQQ0FfMjAxMC0wNy0wMS5jcnQw
// SIG // DAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcD
// SIG // CDANBgkqhkiG9w0BAQsFAAOCAQEAc+6N+7Rbw8FOmN9h
// SIG // o+sAogEspyWNPj5idZtuAa+ZdTw68hQMGSS/yA0YYdE7
// SIG // kNLJJoIBEjOCfbIiF4CqHobAzbIqt9vh5UJg97UJOUKx
// SIG // 5LlM6/5Of/3mZeP43FOq+42auGAJWvQJDtvfGgpzANxB
// SIG // uDtOZ6sOBsi/aTtwSpthtT8Hcy1JfxmON/RmeB0qhfQl
// SIG // iQAQNtlyE6tGJS0Mki16A8pk9/oKN4diOuYrC9M5ULO/
// SIG // eVbS7KHXJv84N5Ef5WoQ1IcJugWISKr0qkow6l6TVW9C
// SIG // GYjYptOVG8rzr2CPU3C5QcfxzdZe7gDRfX4IGZTy3SC9
// SIG // 398WVC/DTi94paH3zqGCA3kwggJhAgEBMIHjoYG5pIG2
// SIG // MIGzMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRN
// SIG // T1BSMScwJQYDVQQLEx5uQ2lwaGVyIERTRSBFU046QjFC
// SIG // Ny1GNjdGLUZFQzIxJTAjBgNVBAMTHE1pY3Jvc29mdCBU
// SIG // aW1lLVN0YW1wIFNlcnZpY2WiJQoBATAJBgUrDgMCGgUA
// SIG // AxUAOrrfkyhl5HrT56P24qdEbliqU9KggcIwgb+kgbww
// SIG // gbkxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5n
// SIG // dG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
// SIG // aWNyb3NvZnQgQ29ycG9yYXRpb24xDTALBgNVBAsTBE1P
// SIG // UFIxJzAlBgNVBAsTHm5DaXBoZXIgTlRTIEVTTjo0REU5
// SIG // LTBDNUUtM0UwOTErMCkGA1UEAxMiTWljcm9zb2Z0IFRp
// SIG // bWUgU291cmNlIE1hc3RlciBDbG9jazANBgkqhkiG9w0B
// SIG // AQUFAAIFANxFzHQwIhgPMjAxNzAyMDgxNjU3MjRaGA8y
// SIG // MDE3MDIwOTE2NTcyNFowdzA9BgorBgEEAYRZCgQBMS8w
// SIG // LTAKAgUA3EXMdAIBADAKAgEAAgIU0wIB/zAHAgEAAgIb
// SIG // OjAKAgUA3Ecd9AIBADA2BgorBgEEAYRZCgQCMSgwJjAM
// SIG // BgorBgEEAYRZCgMBoAowCAIBAAIDFuNgoQowCAIBAAID
// SIG // B6EgMA0GCSqGSIb3DQEBBQUAA4IBAQBYgqYOxhjzFHzf
// SIG // Kw5/yAuSb+KEDYtHuoCwKJ4oWiiwjMSwWVhdYPAYUhyo
// SIG // Uy7rgMJnJ2t/Q0azkRC6235h1GSqgPCnIWjWtSjv9dWR
// SIG // R0twgmXESu+x/KGqs13ACVDZspL7PVHKoPj037IFajym
// SIG // AjsbyYRu8LlQI3LoRZ/cjWar2LzqgxS1yzhn269l1Jne
// SIG // J+RrGO+2WOVDN+qjtQzLVt1tbJVkTgVA0Dxop03hFk01
// SIG // CPpOlHS/EGelN18a30GYzDD8BhGESqQrQPqL3kM8Icta
// SIG // SGEbGNepbmprvKEpagyAml7VmQXi+aS10f7vYFofgTyp
// SIG // IWrGpjAxr8DSWX0Y8a/nMYIC9TCCAvECAQEwgZMwfDEL
// SIG // MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24x
// SIG // EDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9z
// SIG // b2Z0IFRpbWUtU3RhbXAgUENBIDIwMTACEzMAAACxcRN5
// SIG // 33X2NcgAAAAAALEwDQYJYIZIAWUDBAIBBQCgggEyMBoG
// SIG // CSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG
// SIG // 9w0BCQQxIgQg1ZMXyTIeD/REwMyd9MX+9igDmPRq6EXL
// SIG // 2lHlB+nVlsMwgeIGCyqGSIb3DQEJEAIMMYHSMIHPMIHM
// SIG // MIGxBBQ6ut+TKGXketPno/bip0RuWKpT0jCBmDCBgKR+
// SIG // MHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5n
// SIG // dG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
// SIG // aWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1p
// SIG // Y3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAA
// SIG // sXETed919jXIAAAAAACxMBYEFFYXL3ClSZDH/LmDxBBe
// SIG // zKo6KuEHMA0GCSqGSIb3DQEBCwUABIIBAJffU8Zhhr97
// SIG // kS6r1fAZtLL8TbPk0gCn7BsAqd5r06inbrmVCl4NpyYr
// SIG // g5XKapyP0a8FEy52/Gn9CYQNwrKDn+oUlkcDI36jEUjZ
// SIG // tzsetEc4vHFheBZFaDBJ6MBP7JrP+8m1KRI5hF7fC/5A
// SIG // 4bNTmYnSLvnDLN/YrHXubXBmw6712bpyVDaU1yoVWzFL
// SIG // VeydZfSkTDIsowjQASyIz5S0YoDH2UCdyNnC6ZJAN5Ut
// SIG // fY0PsbtSXY+OLOHXoalQ+WqB1tfrSRn4UTdzRKWjKcTg
// SIG // jHyS+IWssOYPQ/cjN1veQqcQzCeSJtbHi7FpDPkL9Ulf
// SIG // Pijwnu2ra1miTlIcPQ7KI6o=
// SIG // End signature block
